/**
 *  Created by hassan.raza on 04/10/2024.
 */

'use strict';

const path = 'public/uploads';
const fs = require('fs');
const multer = require('multer'),
  config = require('../config');
const { cloud } = require('../config/cloud');
const sharp = require('sharp');


const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    try {
      fs.exists(path, function (exists) {
        if (exists) cb(null, path);
        else {
          fs.mkdirSync(path, { recursive: true });
          cb(null, path);
        }
      });
    } catch (error) {
      console.log(error);
    }
  },

  filename: (req, file, cb) => {
    let filename = file.originalname.substr(
      0,
      file.originalname.lastIndexOf('.')
    );
    filename = filename
      .split(',')
      .join('')
      .split('.')
      .join('')
      .split(' ')
      .join('');
    let fileType = file.originalname.substr(
      file.originalname.lastIndexOf('.') + 1
    );
    fileType = fileType
      ? fileType
      : file.mimetype.substr(file.mimetype.lastIndexOf('/') + 1);
    cb(null, filename.toLowerCase() + '-' + Date.now());
  }
}),
  allow_filesType = {
    image: ['image/png', 'image/jpeg', 'image/jpg']
  };

let checkFileType = (file, types) => {
  var found = false;
  for (var i = 0; i < types.length; i++) {
    found = file.mimetype === types[i];
    if (found) {
      break;
    }
  }
  return found;
};

exports.upload = multer({
  storage: storage,
  fileFilter: (req, file, cb) => {
    if (!file.mimetype.startsWith('image'))
      return cb({
        message: 'Invalid file type please check allowed files in api docs.'
      });
    cb(null, true);
  }
});

const removeFileFromTmpFolder = async (filepath) => {
  return await fs.promises.unlink(filepath);
}


const resizeImage = async (inputPath, outputPath, extention = 'jpeg', width = 800, height = 800, quality = 90) => {
  try {
    await sharp(inputPath)
      .resize(width, height, {
        fit: sharp.fit.inside,
        withoutEnlargement: true
      })
      .toFormat(extention, { quality })
      .toFile(outputPath);

    return outputPath;
  } catch (error) {
    console.error('Error resizing image:', error);
  }
};

exports.prepareAndUpload = async (req, res, next) => {

  if (req.file || req.files) {
    try {
      let filePath = req.file.path;
      let extention = req.file.mimetype.substr(req.file.mimetype.lastIndexOf('/') + 1)
      let overSized = req.file.size > 1000000


      if (overSized)
        filePath = await resizeImage(filePath, `${filePath}-compressed.${extention}`, extention)

      await cloud.uploader.upload(filePath, {
        public_id: req.file.filename
      })

      await removeFileFromTmpFolder(filePath)

    } catch (error) {
      console.log(error);
    }

  }

  next();
}

exports.baseUrlGenerator = req => {
  if (req.file) {
    //for single file
    return {
      url: cloud.url(req.file.filename, {
        fetch_format: 'auto',
        quality: 'auto'
      }),
      size: req.file.size,
      filename: req.file.filename,
      type: req.file.mimetype
    };
  }
};
