/* eslint-disable prefer-const */
/** @format */

const _ = require('lodash');
const mongoose = require('mongoose');
const { ObjectId } = require('mongoose').Types;
const {
  _basePost,
  _baseFetch,
  _basePut,
  _baseRemove
} = require('./util-generic-functions');
const { _responseWrapper } = require('./util-response');

const validateObjId = id => ObjectId.isValid(id);
const handleFiles = (files, body) => {
  body.media = files.media;
  return body;
};

const excludePut = ['__v', 'id', 'createdBy', 'media'];
const excludePost = ['__v', 'id'];

// eslint-disable-next-line no-return-assign
const contAggQuery = (agg, _id) => {
  if (agg[0].$match) agg[0].$match._id = new mongoose.Types.ObjectId(_id);
  else agg.unshift({ $match: { _id: new mongoose.Types.ObjectId(_id) } });
  return agg;
};
// creating the controller
exports.createDoc = (Model, rest = {}) => async req => {
  // here we are getting all the keys of our schema object so I exclude (__v, Id, and _id)

  try {
    // Validating the incomming req body from schema
    const { returnDoc, populate = {}, insertMany, validateFN = () => { } } = rest;
    const { error } = validateFN(req.body);
    //return error response if invalid request or something
    if (error) return _responseWrapper(true, error.details[0].message, 400);
    // return { status: false, error: { message: error.details[0].message } };

    const varToPick = _.keys(Model.schema.tree).filter(
      el => !excludePost.includes(el)
    );
    const { body } = req;
    body.createdBy = req.user && req.user._id;
    // eslint-disable-next-line no-use-before-define
    if (req.files) req.body = handleFiles(req.files, req.body);
    // Product out data and get only those fields which is in the schema object

    // if (insertMany)
    // doc = await _basePost(Model, body, insertMany);
    // else
    let doc = await _basePost(Model, _.pick(body, varToPick), insertMany);

    if (doc.status && Object.keys(populate).length) {
      let pop = await Model.populate(doc.data, populate);
      if (pop) {
        doc.data = pop
      }
    }

    if (returnDoc) return doc;
    return this.errOrSuccessHandler(req, Model, 'createSuccess', doc);
  } catch (error) {
    console.log(error);
    //return this.errOrSuccessHandler(req, Model, "error", doc);
  }
};

// *? Getting Single/All the currency controller
exports.getDocs = (Model, rest = {}) => async req => {
  let {
    params: { id, slug },
    query: { skip = 0, limit = 20 }
  } = req;

  const {
    agg,
    query = {},
    populate,
    returnDoc,
    select = '-__v -isDeleted ',
    sort = { createdAt: -1 },
    single = false
  } = {
    ...rest
  };

  let { findMethod } = rest;
  const args = {
    query: agg || { isDeleted: false, ...query },
    extra: { skip: +skip, limit: +limit },
    parameterToGet: select,
    populate,
    sort
    // popParamToGet: populate && populate.params,
  };

  //set method type
  if (!findMethod)
    findMethod =
      (agg && 'Aggregate') ||
      (populate && 'FindWithPopulate') ||
      'FindWithCount';

  if (id || slug) {
    if (id) {
      const isValid = validateObjId(id);
      if (!isValid) return _responseWrapper(false, 'invalidId', 400);
      // contAggQuery => this func just edit/add the $match object with current ObjectId
      id = new ObjectId(id);
      if (agg) args.query = contAggQuery(agg, id);
      if (id) args.query._id = id;
    }

    if (slug) args.query.slug = slug;
    findMethod = (populate && (id || slug) && 'FindOneWithRefs') || 'FindOne';
  }
  if (single) {
    args.extra.skip = 0;
  }
  const doc = await _baseFetch(Model, args, findMethod);

  if (returnDoc) return doc;
  return this.errOrSuccessHandler(null, null, 'fetchSuccess', doc);
};

// creating the controller
// agg => aggregation
exports.updateDoc = (Model, params = {}) => async req => {
  const {
    params: { id },
    body,
    files
  } = req;
  const { updateObject = {}, returnDoc, populate } = params;

  const { _id } = updateObject.query || {};

  const isValid = validateObjId(id || _id);

  if (!isValid) return _responseWrapper(false, 'invalidId', 400);

  if (files) req.body = handleFiles(files, body);

  const varToPick = _.keys(Model.schema.tree).filter(
    el => !excludePut.includes(el)
  );

  //  if schema contain lastUpdated field, then will update that field with current date/time
  if (varToPick.includes('lastUpdated')) body.lastUpdated = new Date();

  const query = updateObject.query ? updateObject.query : {};
  let isDeleted = varToPick.includes('isDeleted');
  isDeleted = isDeleted ? { isDeleted: false } : {};

  // creating object which is contain which one to update, what to update
  const args = {
    query: { _id: new ObjectId(id), ...isDeleted, ...query },
    updateObject: !_.isEmpty(updateObject)
      ? updateObject.update
      : _.pick(body, [...varToPick, "$push", "$inc", "$pull", "$set"]),
    parameterToGet: '-__v'
  };

  args.populate = populate || '';
  const doc = await _basePut(Model, args, 'findOneAndUpdate');

  // if (doc.status)
  //   createActivityLog(
  //     req,
  //     Model.modelName,
  //     `${Model.modelName} is updated @ SALES APP`
  //   );
  if (returnDoc) return doc;
  return this.errOrSuccessHandler(req, Model, 'updateSuccess', doc);
};

// creating the controller
exports.deleteDoc = (Model, removeType) => async req => {
  const {
    params: { id }
  } = req;

  const isValid = validateObjId(id);
  if (!isValid) return _responseWrapper(false, 'invalidId', 400);

  const args = { query: { _id: id }, removeObject: { isDeleted: true } };

  const doc = await _baseRemove(Model, args, removeType);

  // if (returnDoc) return doc;
  return this.errOrSuccessHandler(req, Model, 'deleteSuccess', doc);
};

exports.errOrSuccessHandler = (req, Model, msg, doc) => {
  let statusCode;
  let action;
  // console.log('action', action);

  // check for duplicate
  if (!doc.status && doc.error.code === 11000) return _responseWrapper(false, doc.error.message, 409);

  // !check if any error
  if (!doc.status) return _responseWrapper(false, doc.error.message, 400);

  //! check if data is null
  if (!doc.data && !doc.error) return _responseWrapper(false, 'notFound', 404);


  switch (msg) {
    case 'fetchSuccess':
      statusCode = 200;
      break;
    case 'createSuccess':
      statusCode = 201;
      action = 'Created';
      break;
    case 'updateSuccess':
      statusCode = 200;
      action = 'Updated';
      break;
    case 'deleteSuccess':
      statusCode = 200;
      action = 'Deleted';
      break;
    case 'error':
      statusCode = 204;
      //action = "Deleted";
      break;
    default:
      statusCode = 500;
  }

  // if (action)
  // ! it does not run on get request
  // createActivityLog(
  //   req,
  //   `${Model.modelName} ${action}`,
  //   `${Model.modelName} ${action} @ IAWCI`,
  //   {}
  // );

  //* if everything is okay, then sending the response with data
  // doc = _dataOmit(doc, ["__v", "isDeleted"]);
  return _responseWrapper(false, msg, statusCode, doc);
};

// for (const key in files) {
//   if (_.has(files, key)) {
//     // const element = files[key];
//     // const [outFromArray] = element;
//     const key = files.fileUpload[0].mimetype.split("/")[0];
//     body.media = {};
//     key === "video"
//       ? (body.media[key] = files)
//       : (body.media[key] = [{ ...files }]);
//   }
// }
