/* eslint-disable no-restricted-syntax */
/** @format */

const mongoose = require('mongoose');

const { convert } = require('joigoose')(mongoose);
const crypto = require('crypto');
const Joi = require('joi');
const bcrypt = require('bcrypt');
const baseSchema = require('../../baseSchema');
const { isValidEmail } = require('../../../utils/email');

const userJoigooseSchema = Joi.object({
  name: Joi.string()
    .required()
    .min(1),
  email: Joi.string()
    .email()
    .required(),
  avatar: Joi.object({
    type: Joi.string(),
    size: Joi.number(),
    filename: Joi.string(),
    url: Joi.string()
  }),
  password: Joi.string()
    .required()
    .min(6),
  confirmPassword: Joi.string().required(),
  passwordResetToken: Joi.string(),
  passwordResetExpires: Joi.date(),
  passwordChangeAt: Joi.date(),
  location: Joi.object({
    country: Joi.string().allow(''),
    city: Joi.string().allow(''),
    state: Joi.string().allow(''),
    postalCode: Joi.string().allow(''),
    address: Joi.string().allow('')
  }),
  followings: Joi.array()
    .items(Joi.string()),
  followers: Joi.array()
    .items(Joi.string()),
  savedPosts: Joi.array()
    .items(Joi.string()),
  visibility: Joi.string().default('public').valid('public', 'private'),
  role: Joi.string().default('user').valid('user', 'artist'),
  verified: Joi.boolean().default(false),
  status: Joi.boolean().default(true),
  ...baseSchema
});

let userSchema = convert(userJoigooseSchema);
userSchema.email.unique = true;
userSchema.email.lowercase = true;
userSchema.password.select = false;

userSchema.followings = [{
  type: mongoose.Schema.Types.ObjectId,
  ref: 'User'
}];
userSchema.followers = [{
  type: mongoose.Schema.Types.ObjectId,
  ref: 'User'
}];

userSchema.savedPosts = [{
  type: mongoose.Schema.Types.ObjectId,
  ref: 'Post'
}];

userSchema = new mongoose.Schema(userSchema);

// comparing/verify user who want to login password and db password are same or not
userSchema.methods.verifyPassword = async function (candidatePwd, dbPwd) {
  return await bcrypt.compare(candidatePwd, dbPwd);
};

// when password change so create this field in db
userSchema.pre('save', async function (next) {
  if (this.isModified('password') || !this.isNew) return next();
  this.passwordChangeAt = Date.now() - 1000;
});


// creating the hash of password
userSchema.pre('save', async function (next) {
  if (!this.isModified('password')) return next();

  this.password = await bcrypt.hash(this.password, 12);
  this.confirmPassword = undefined;

  next();
});

// generating the password reset token for resetting the forgotten password
userSchema.methods.generatePwdResetToken = async function () {
  const resetToken = crypto.randomBytes(32).toString('hex');

  this.passwordResetToken = crypto
    .createHash('sha256')
    .update(resetToken)
    .digest('hex');

  this.passwordResetExpires = Date.now() + 60 * 1000 * 20;
  return resetToken;
};

// check if the user changes his/her password after the token is issued
userSchema.methods.changePasswordAfter = function (jwtIssueDate) {
  return this.passwordChangeAt < jwtIssueDate;
};

// validate the password and confirm password should be same
userSchema.path('password').validate(function (password) {
  return password === this.confirmPassword;
}, `({VALUE}) password and confirm password are not matched`);

userSchema.path('email').validate(function (email) {
  return isValidEmail(email);
}, '({VALUE}) is not a valid email');
// create the Instance of user
const User = mongoose.model('User', userSchema);

// validate the user input by passing it to joigoose schema
const validateUser = userData => userJoigooseSchema.validate(userData);

module.exports = {
  User,
  validateUser
};
