/* eslint-disable no-restricted-syntax */
/** @format */

const mongoose = require('mongoose');

const { convert } = require('joigoose')(mongoose);
const Joi = require('joi');
const baseSchema = require('../../baseSchema');

const storyJoigooseSchema = Joi.object({
  user: Joi.string()
    .required(),
  caption: Joi.string()
    .optional()
    .max(2200),
  media: Joi.array()
    .items(Joi.object({
      type: Joi.string(),
      size: Joi.number(),
      filename: Joi.string(),
      url: Joi.string()
    })),
  expiry: Joi.date().required(),
  watchedBy: Joi.array().items(Joi.object({
    user: Joi.string().required(),  // User who watched the status
    watchedAt: Joi.date().required()  // Time when the user watched the status
  })),
  ...baseSchema
});

let storySchema = convert(storyJoigooseSchema);

storySchema.user = {
  type: mongoose.Schema.Types.ObjectId,
  ref: 'User'
};

storySchema.watchedBy = [{
  user: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true
  },
  watchedAt: {
    type: Date,
    required: true
  }
}];


storySchema = new mongoose.Schema(storySchema);


// Define methods for the Story schema

// Method to check if the story has expired
storySchema.methods.isExpired = function () {
  return Date.now() > this.expiry;
};

// Method to update the expiry date
storySchema.methods.extendExpiry = function (newExpiryDate) {
  this.expiry = newExpiryDate;
  return this.save();
};

// Method to mark a story as watched by a user
storySchema.methods.markAsWatched = async function (userId) {
  const alreadyWatched = this.watchedBy.find(entry => entry.user.equals(userId));

  if (!alreadyWatched) {
    this.watchedBy.push({
      user: userId,
      watchedAt: Date.now()
    });
    await this.save();
  }
};

// Method to get all users who have watched the story
storySchema.methods.getWatchedByUsers = function () {
  return this.watchedBy;
};



const Story = mongoose.model('Story', storySchema);



// validate the story input by passing it to joigoose schema
const validateStory = storyData => storyJoigooseSchema.validate(storyData);

module.exports = {
  Story,
  validateStory
};
